#!/bin/bash 
#CoM====================================================================
#CoM...dq_modes.sh: Generates conformers from normal mode amplitudes.
#CoM--------------------------------------------------------------------
#CoM
#CoM   Main goal: 
#CoM   ++++++++++
#CoM
#CoM  *To generate series of model conformers.
#CoM
#CoM   Arguments: 
#CoM   ++++++++++
#CoM
#CoM  *Choice (-choice):
#CoM   ALL: All amplitudes are taken into account simultaneously.
#CoM   FIRST: First one, then two first ones, three first ones, etc.
#CoM
#CoM   Required:  
#CoM   =========
#CoM
#CoM  *A normal mode file in CERFACS format, i.e.:
#CoM
#CoM   VECTOR
#CoM   ------
#CoM   x-vect y-vect z-vect 
#CoM   x-vect y-vect z-vect 
#CoM   ...
#CoM   VECTOR
#CoM   ------
#CoM   x-vect y-vect z-vect 
#CoM   ...
#CoM
#CoM  *A coordinate file in PDB or free format, i.e.:
#CoM   x y z 
#CoM   or:
#CoM   x y z mass
#CoM
#CoM   Output:
#CoM   +++++++
#CoM
#CoM  *Conformers named: "coordinate-file-prefix"_conf"number".pdb;
#CoM   or a multiple PDB RMN-like file (-output).
#CoM
#CoM   Properties:
#CoM   +++++++++++
#CoM
#CoM  *An amplitude file is generated by the script, 
#CoM   if vector numbers (-vect) are provided.
#CoM   Otherwise, at least one amplitude file (-fil) is required.
#CoM
#CoM  *Amplitude files:
#CoM   First word should give the mode number, second, its amplitude.
#CoM   A third word can provide a range for amplitude randomization.
#CoM   No more than three words per line.
#CoM   Goal: to recognize wrong amplitude files.
#CoM
#CoM  *Uses fortran program dq_modes (see version number below).
#CoM
#CoM  *The fortran program is copied and compiled in current directory.
#CoM   Purpose: to be able to reproduce the analysis in the future.
#CoM
#CoM   Attention: 
#CoM   ++++++++++
#CoM
#CoM  *Uses temporary files. 
#CoM   So, do NOT run this script twice at a time in same directory.
#CoM
#CoM--------------------------------------------------------------------
#CoM   This software is released under the CeCILL FREE SOFTWARE LICENSE.
#CoM   In short: this license is compatible with the GNU GPL.
#CoM   Specifically: it grants users the right to modify and redistribute 
#CoM   this software within the framework of an open source distribution 
#CoM   model. The complete text of the license can be found there:
#CoM   http://www.cecill.info/licences/Licence_CeCILL_V2-en.html
#CoM-----------------------------------------------------------------------
#CoM   Please mail bug reports to yves-henri.sanejouand at univ-nantes(fr).
#ChRo  v1.00: YHS-Apr-2016, Nantes.
pgmprf='Dqmodes'

# Script version:
version='v1.9, April 2017.'

# Fortran program used hereafter:
# -------------------------------
fornam='dq_modes'

# Version used (file fornam.f_forvrs):
# ------------------------------------
forvrs='1.47'

# Default location (path) of fortran programs:
# --------------------------------------------
pgmdir=~/inputs/shell/fortran

# Default fortran compiler (for instance, under Ubuntu: gfortran):
# ----------------------------------------------------------------
gfor=gfortran

pgmnam='+'$pgmprf'> '
pgmwrn='%'$pgmprf'-Wn> '
pgmerr='%'$pgmprf'-Er> '

# To avoid problems, noteworthy with sort -g:
# (to specify the meaning of a dot)
export LC_ALL=C

echo $pgmnam 'Generates conformers from sets of normal mode amplitudes.'
echo $pgmnam $version

# Arguments and options are read:
# -------------------------------
echo $pgmnam $version | sed 's/ //g' > dqmod.amplitudes

# Defauts:
ntrydef=YES
pdbref=NONE
modes=NONE
mpdb=NONE
amplv=10.0
fluct=0.0
nparv=0
chrd=ALL
nmod=-1
nvecrd=0
ivec=-1
rcut=0
dhalf=-1
kenm=1
mini=0
ntry=1
fconf=1
prtl=0
seed=-1
debrd=NO
# une liste de fichiers pour finir:
lstfil=NONE

if [ $# -gt 0 ] 
then
help=OFF
nargrd=$#
nargs=0
while [ $nargs -lt $nargrd ]
do
# Un - en premier caractere...
  cmd=`echo '+'$1 | tr '[A-Z]' '[a-z]' | sed 's/+//'`
  case $cmd in 
 '-h'|'-help') 
   help=ON 
   nargs=$(( $nargs + 1 ))
  ;;
  *)
  if [ $# -ge 2 ] ; then
  nargs=$(( $nargs + 2 ))
  fcararg=`echo $2 | cut -c1`
  if [ $fcararg = \- ] ; then
  case $2 in
  *[!\-,!\.,!0-9]*)
  echo ' '
  echo $pgmerr $cmd 'must be followed by a parameter (or by a number).'
  lstfil=FOUND
  nargs=$nargrd
  help=ON
  esac
  fi
  case $cmd in
 '-f'|'-fil'|'-file'|'-files'|'-filename'|'-filenames')   
  nfil=$(( $# - 1 ))
  nargs=$nargrd
  lstfil=FOUND
  ;;
 '-a'|'-amp'|'-ampl'|'-amplitude') amplv=$2  ;;
 '-cho'|'-choix'|'-choice') chrd=$2 ;;
 '-com'|'-comp'|'-compil'|'-compiler') gfor=$2 ;;
 '-coo'|'-coor'|'-coordinate'|'-coordinates') pdbref=$2 ;;
 '-cut'|'-cuto'|'-cutof'|'-cutoff') rcut=$2 ;;
 '-d'|'-deb'|'-debug') debrd=$2 ;;
 '-for'|'-fort'|'-fortran') pgmdir=$2 ;;
 '-fir'|'-firs'|'-first') fconf=$2 ;;
 '-flu'|'-fluc'|'-fluct'|'-fluctuation'|'-fluctuations') fluct=$2 ;;
 '-min'|'-mini'|'-minimisation'|'-minimization') mini=$2 ;;
 '-mod'|'-mode'|'-modes') modes=$2 ;;
 '-n'|'-num'|'-number') nmod=$2 ;;
 '-o'|'-out'|'-output') mpdb=$2 ;;
 '-pdb') pdbref=$2 ;;
 '-prt'|'-pri'|'-print'|'-printing') prtl=$2 ;;
 '-r'|'-ran'|'-rang'|'-range') dhalf=$2 ;;
 '-see'|'-seed')   seed=$2 ;;
 '-sou'|'-source') pgmdir=$2 ;;
 '-spr'|'-spring') kenm=$2 ;;
 '-t'|'-try'|'-tri'|'-trial'|'-trials') 
  ntrydef=NO
  ntry=$2 
  ;;
 '-v'|'-vec'|'-vect'|'-vector') 
  nvecrd=$(( $nvecrd + 1 ))
  [[ $( echo "$fluct <= 0.0" | bc ) -eq 1 ]] && fluct=0.0
  [[ $ivec -gt 0 ]] && echo $ivec $amplv $fluct >> dqmod.amplitudes
  ivec=$2 
  ;;
  *) 
  echo ' '
  echo $pgmerr $cmd 'is not an expected option.'
  lstfil=FOUND
  nargs=$nargrd
  help=ON
  ;;
  esac
# Argument suivant:
  if [ $lstfil = NONE ] ; then shift ; fi
  else
  echo ' '
  echo $pgmerr $cmd 'is the last argument ?'
  nargs=$(( $nargs + 1 ))
  help=ON
  fi
  ;;
  esac
# Argument suivant:
  shift 
done
# Pas d'argument:
else
help=ON
fi
[[ $( echo "$fluct <= 0.0" | bc ) -eq 1 ]] && fluct=0.0
[[ $nvecrd -gt 0 ]] && echo $ivec $amplv $fluct >> dqmod.amplitudes

ampl=NONE
nargs=$#
if [ $nargs -gt 0 ] ; then
# Files among arguments left ? (robust syntax ?)
if [ `ls $* 2>&1 | grep -c 'ls:'` -ne 0 ] ; then
ampl=PB
[[ $nargs -eq 1 ]] && ampl=`echo $1 | tr '[a-z]' '[A-Z]'`
if [ $ampl != NONE ] ; then
   echo ' '
   echo $pgmerr 'Last arguments are ALL expected to be valid filename(s).'
   help=ON
fi
fi
else
nampf=$( wc -l dqmod.amplitudes | awk '{ print $1 }' )
if [ $nampf -eq 1 ] ; then
if [ $help != ON ] ; then
   echo ' '
   echo $pgmerr 'No amplitude file (-f) and no vector (-vect) index provided. Expecting one of them.'
   help=ON
fi
else
   [[ $ntrydef = YES ]] && ntry=10
   if [ $ntry -gt 1 ] ; then
      nfluct=$( awk '{ if ( NR > 1 && $3 > 0.0 ) print }' dqmod.amplitudes | wc -l )
      if [ $nfluct -eq 0 ] ; then
         echo $pgmwrn $ntry 'trials (-try) but no mode fluctuation (-fluct) provided. This is fixed.'
         cp dqmod.amplitudes dqmod.amplitudes_init
         awk '{ if ( NR > 1 ) { print $1, $2, $2 } else print }' dqmod.amplitudes_init > dqmod.amplitudes
         rm -f dqmod.amplitudes_init
      fi
   fi
   ampl=BUILT
   nargs=1
fi
fi
[[ $ampl != BUILT ]] && rm -f dqmod.amplitudes

debug=`echo $debrd | tr '[a-z]' '[A-Z]'`
choix=`echo $chrd | tr '[a-z]' '[A-Z]'`
pgmout=`echo $pgmprf | tr '[A-Z]' '[a-z]'`

if [ $help = ON ] ; then
echo ' '
echo ' Action: Generates conformers from sets of normal mode amplitudes.'
echo ' Usage : '$pgmout'.sh { arguments } { options } -f filename(s)'      
echo ' '
echo ' After arguments and options:'
echo '-files   amplitude-filename(s)                                                   (default:' $ampl')'
echo ' '
echo ' Arguments:'
echo '-coor    coordinate-filename                                                     (no default)'
echo '-modes   vector-filename                                                         (no default)'
echo ' '
echo ' Main options:'
echo '-vector  index-to-be-used-when-there-is-no-amplitude-file                        (default:' $ivec')'
echo '-amplitude to-be-used-when-there-is-no-amplitude-file                            (default:' $amplv')'
echo '-fluctuation to-be-added-to-amplitude-when-there-is-no-amplitude-file            (default:' $fluct')'
echo '-choice  of-modes-in-amplitude-file                                              (default:' $choix')'
echo '-try     number-of-trials-per-amplitude-file                                     (default:' $ntry')'
echo '-number  number-of-mode-amplitude(s)-picked-at-each-trial                        (default:' $nmod')'
echo '-mini    number-of-minimization-steps-in-complementary-space                     (default:' $mini')'
echo '-cutoff  for-computing-elastic-network-energy-to-be-minimized                    (default:' $rcut')'
echo '-output  RMN-like-PDB-filename-with-all-generated-conformers                     (default:' $mpdb')'
echo ' '
echo ' More options:'
echo '-seed    for-random-choices                                                      (default:' $seed')'
echo '-first   first-conformer-index-on-output                                         (default:' $fconf')'
echo '-spring  force-constant-of-the-elastic-network                                   (default:' $kenm')'
echo '-range   for-a-decaying-force-constant-of-the-elastic-network                    (default:' $dhalf')'
echo '-compile fortran-compiler                                                        (default:' $gfor')'
echo '-source  path-to-fortran-source                                                  (default:' $pgmdir')'
echo '-print   level-for-outputs                                                       (default:' $prtl')'
echo '-debug   to-keep-most-temporary-files                                            (default:' $debug')'
echo ' '
exit
fi

# Checks:
# =======
notf=0

# Les fichiers en input existent TOUS ?
if [ $pdbref != NONE ] ; then
if [ -f $pdbref ] ; then 
  natpdb=$( awk '{ if ( $1 == "ATOM" || substr($1,1,6) == "HETATM" ) print }' $pdbref | wc -l )
  natca=$( awk '{ if ( $1 == "ATOM" && $3 == "CA" && $4 != "CA" && $4 != "CAL" ) print }' $pdbref | wc -l )
else
  echo ' '
  echo $pgmerr 'Coordinate file "'$pdbref'" not found. Mandatory.' 
  notf=`expr $notf + 1`
fi
else
  echo ' '
  echo $pgmerr 'No coordinate filename (-coor). Mandatory.'
  notf=`expr $notf + 1`
fi

if [ $modes != NONE ] ; then
if [ -f $modes ] ; then 
nvec=`grep -i vector $modes | grep -ci value`
if [ $nvec -eq 0 ] ; then
  echo ' '
  echo $pgmerr 'Mode file "'$modes'" is not in CERFACS format. Mandatory.'
  notf=`expr $notf + 1`
fi
else
  echo ' '
  echo $pgmerr 'Mode file "'$modes'" not found. Mandatory.' 
  notf=`expr $notf + 1`
fi
else
  echo ' '
  echo $pgmerr 'No mode filename (-modes). Mandatory.'
  notf=`expr $notf + 1`
fi

# Additional checks:
# ==================

case $ivec in
*[!0-9,!\-]*) 
  echo ' '
  echo $pgmerr '-vector: an integer number is required (mode index).'
  notf=`expr $notf + 1` ;;
esac

case $amplv in
*[!0-9,!\-,\.]*)
  echo ' '
  echo $pgmerr '-amplitude: a real number is required (displacement along a mode).'
  notf=`expr $notf + 1` ;;
esac

case $ntry in
*[!0-9,!\-]*) 
  echo ' '
  echo $pgmerr '-try: an integer number is required (number of trials).'
  notf=`expr $notf + 1` ;;
*)
  if [ $ntry -le 0 ] ; then ntry=1 ; fi
esac

case $nmod in
*[!0-9,!\-]*) 
  echo ' '
  echo $pgmerr '-number: an integer number is required (number of modes).'
  notf=`expr $notf + 1` ;;
esac

case $rcut in
*[!0-9,!\-,\.]*)
  echo ' '
  echo $pgmerr '-cutoff: a real number is required (a distance).'
  notf=`expr $notf + 1` ;;
esac

case $dhalf in
*[!0-9,!\-,\.]*)
  echo ' '
  echo $pgmerr '-range: a real number is required (a distance).'
  notf=`expr $notf + 1` ;;
esac

case $kenm in
*[!0-9,!\-,\.]*)
  echo ' '
  echo $pgmerr '-spring: a real number is required (a force constant).'
  notf=`expr $notf + 1` ;;
esac

case $mini in
*[!0-9,!\-]*) 
  echo ' '
  echo $pgmerr '-mini: an integer number is required (a number of steps).'
  notf=`expr $notf + 1` ;;
*)
  if [ $mini -gt 0 ] ; then
#    By default: ENM standard cutoffs.
     guess=NO
     [[ $( echo "$dhalf <= 0" | bc ) -eq 1 ]] && [[ $( echo "$rcut <= 0" | bc ) -eq 1 ]] && guess=YES
     if [ $guess = YES ] ; then
     if [ $natca -eq $natpdb -a $natpdb -gt 0 ] ; then 
        rcut=12.0
     else
        rcut=5.0
     fi
     fi
  else
     mini=0
  fi
esac

case $fconf in
*[!0-9,!\-]*) 
  echo ' '
  echo $pgmerr '-first: an integer number is required (an index).'
  notf=`expr $notf + 1` ;;
*)
  if [ $fconf -le 0 ] ; then fconf=1 ; fi
esac

case $seed in
*[!0-9,!\-]*) 
  echo ' '
  echo $pgmerr '-seed: an integer number is required.'
  notf=`expr $notf + 1` ;;
esac
if [ $notf -gt 0 ] ; then exit ; fi

case $choix in
 'A'|'AL'|'ALL') choix=ALL ;;
 'F'|'FST'|FIRS|FIRST) choix=FIRST ;;
 'R'|'RAN'|RAND|RANDOM) 
  choix=RANDOM 
  if [ $nmod -le 0 ] ; then nmod=1 ; fi 
  ;;
  *)
  echo $pgmwrn '-choice: "'$choix'" is an unknown way of choosing mode amplitudes.'
  echo $pgmwrn 'Known ones: ALL, FIRST, RANDOM. Assumed: ALL.'
  choix=ALL ;;
esac

case $debug in
 'Y'|YES|O|OUI) 
  debug=YES ;;
 'N'|NO|NON) debug=NO ;;
  *)
  echo $pgmwrn '-debug: "'$debug'" is an unknown option.'
  echo $pgmwrn 'Known ones: YES, NO. Assumed: NO.'
  debug=NO ;;
esac 

chcur=$( echo $mpdb | tr '[a-z]' '[A-Z]' )
case $chcur in
N|NO|NON|NONE) 
mpdb=NONE ;;
*)
if [ -f $mpdb ] ; then
echo $pgmwrn '"'$mpdb'" file already exists. A copy is saved.'
mv $mpdb $mpdb'_old'
fi
mpdbpref=$( basename $mpdb | awk -F'.' '{ print $1 }' )
;;
esac

# Compiling:
# ==========
# Compilation du fortran (si necessaire):

# Full fortran program name:
vforpgm=$fornam'.f_v'$forvrs
[[ ! -f $vforpgm ]] && vforpgm=$pgmdir/$fornam'.f_v'$forvrs

forcur=$fornam'_vcur.f'
execur=$fornam'_vcur'
ndiff=-1
if [ -f $vforpgm -a -f $forcur ] ; then
ndiff=`diff -s $vforpgm $forcur | grep -c '<'`
if [ $ndiff -gt 0 ] ; then
   echo $pgmwrn '"'$forcur'" and "'$vforpgm'" differ. Former one replaced (a copy is saved).'
   cp $forcur $forcur'_old'
   cp $vforpgm $forcur
   rm -f $execur
fi
else
if [ -f $forcur ] ; then
   echo $pgmwrn 'Fortran source "'$vforpgm'" not found.'
   echo $pgmwrn 'Local fortran source to be used without checking if it is the expected one.'
fi
if [ -f $vforpgm ] ; then
   echo $pgmwrn 'Fortran source "'$vforpgm'" copied locally.'
   cp $vforpgm $forcur
   ndiff=0
fi
fi

if [ -x $execur ] ; then
echo $pgmwrn 'Local executable "'$execur'" to be used.'
if [ $ndiff -gt 0 ] ; then 
   echo $pgmwrn 'It is expected to have been compiled using "'$vforpgm'".' 
else
   if [ -f $vforpgm ] ; then echo $pgmwrn 'It may have been compiled using "'$vforpgm'".' ; fi
fi
else
echo $pgmnam 'Where is the fortran compiler ?'
if which $gfor ; then
   echo $pgmnam 'Fortran compilation to be performed locally.'
else
   echo ' '
   echo $pgmerr $gfor 'fortran compiler needed.'
   exit
fi
if [ -f $forcur ] 
then
   vcur=`grep 'version=' $forcur | sed 's/version=//' | sed 's/ Vers/vers/'`
   echo $pgmnam 'To be compiled:' $vcur
   $gfor -o $execur $forcur
else
   echo ' '
   echo $pgmerr 'No local executable.'
   echo $pgmerr 'Fortran source "'$vforpgm'" required.'
   exit
fi
if [ -x $execur ] ; then
   echo $pgmnam 'Local executable' $execur 'built in current directory.'
else
   echo ' '
   echo $pgmerr 'Compilation of' $forcur 'failed.'
   exit
fi
fi

# Inputs:
# =======

cat > dqmod.dat_main << EOFdat 
COORdinate filename  = $pdbref
NORMal mode filename = $modes
AMPLitude filename   = dq_modes.amplitudes
CUTOff for the ENM   = $rcut
RANGe of-force-cst   = $dhalf
FORCe constant       = $kenm
MINImization steps   = $mini
OUTPut filename pref.= conformer
PRINting level       = $prtl
EOFdat

# Outputs:
# ========

if [ $mpdb = NONE ] ; then
   pgmsum=$pgmout'.sum'
else
   pgmsum=$mpdbpref'.sum'
fi
if [ -f $pgmsum ] ; then
  echo $pgmwrn '"'$pgmsum'" already exists. A copy is saved.'
  mv $pgmsum $pgmsum'_old'
fi

# Single column title (for awk, gnuplot, etc):
echo $pgmnam $version | sed 's/ //g' > $pgmsum
echo $pgmnam 'Job started on' `date`', in' $HOST 'directory' `pwd` | sed 's/ //g' >> $pgmsum

ndot=`basename $pdbref | awk -F'.' '{ print NF-1 }'`
if [ $ndot -gt 0 ] ; then 
   prefout=`basename $pdbref | awk -F'.' '{ print $1 }'`  
else
   prefout=`basename $pdbref`
fi
if [ $mini -gt 0 ] ; then prefout=$prefout'_min' ; fi

# =======================
# Loop on input files(s):
# =======================

echo ' '
echo $pgmnam 'Conformer generation:'

nargrd=$nargs
nargs=0
nampl=0
nred=0
nconf=$(( $fconf - 1 ))
while [ $nargs -lt $nargrd ] ; do
if [ $ampl = BUILT ] ; then
amplfil=dqmod.amplitudes
else
amplfil=$1
echo $pgmnam 'Amplitude file: "'$amplfil'".'
fi
if [ -f $amplfil ] ; then
# Possible title lines are removed:
awk '{ if ( substr($1,1,1) != "!" && substr($1,1,1) != "#" ) print }' $amplfil | grep -v '[A-Z,a-z]' > dq_modes.amplitudes_all
nl=`wc -l dq_modes.amplitudes_all | awk '{ print $1 }'`

# Current amplitude file:
if [ $nl -gt 0 ] ; then
nwmin=`awk '{ print NF }' dq_modes.amplitudes_all | sort -n | head -1`
nwmax=`awk '{ print NF }' dq_modes.amplitudes_all | sort -n | tail -1`
# Is it as expected ?
pb=0
if [ $nwmin -lt 2 ] ; then
   echo $pgmwrn 'Amplitude files have at least two columns (mode-number q).'
   pb=$(( $pb + 1 ))
fi
if [ $nwmax -gt 3 ] ; then
   echo $pgmwrn 'Amplitude files have at most three columns (mode-number q-min dq-max).'
   pb=$(( $pb + 1 ))
fi
if [ $nwmin -eq $nwmax ] ; then
   nwampl=$nwmin
else
   nwampl=$nwmax
fi
# Adding fluctuations, if necessary:
if [ $nwampl -eq 2 -a $ntry -gt 1 ] ; then
if [ $( echo "$fluct > 0.0" | bc ) -eq 1 ] ; then
   cp dq_modes.amplitudes_all dq_modes.amplitudes_all1
   awk '{ print $1, $2, $fluct }' dq_modes.amplitudes_all1 > dq_modes.amplitudes_all
   rm -f dq_modes.amplitudes_all1
   nwampl=3
fi
fi
if [ $pb -gt 0 ] ; then nl=0 ; fi
fi

if [ $nl -gt 0 ] ; then
nampl=$(( $nampl + 1 ))

# Number of modes used at a time:
npick=$nl
if [ $choix = RANDOM ] ; then
if [ $nmod -lt $nl ] ; then npick=$nmod ; fi
if [ $nmod -ge $nl ] ; then npick=$nl ; fi
if [ $seed -lt 1 -o $seed -gt 2147483647 ] ; then
   seed=$( awk 'BEGIN{srand(); print rand()*2147483647+1}' | awk -F '.' '{ print $1 }' )
   echo $pgmwrn '-seed: set to' $seed 
fi
else
# Useless (avoid empty string):
seed=31416
fi

# Several trials:
# ---------------

# Actual number of trials:
if [ $nwampl -eq 2 ] ; then 
[[ $ntry -gt 1 ]] && nred=$(( $nred + 1 ))
ntrycur=1  
else
ntrycur=$ntry
fi

case $choix in
FIRST)  
if [ $nmod -gt 0 -a $nmod -lt $nl ] ; then
ntrycur=$nmod
else
ntrycur=$nl 
fi ;;
RANDOM) ntrycur=$ntry ;;
esac

try=0
while [ $try -lt $ntrycur ] 
do
try=$(( $try + 1 ))
if [ $choix = FIRST ] ; then npick=$try ; fi
echo $pgmnam 'Conformer' $try 'generated with' $npick 'mode(s).'

# Choice of modes taken into account:
rm -f dq_modes.amplitudes

# All of them:
if [ $npick -eq $nl ] ; then
cp dq_modes.amplitudes_all dq_modes.amplitudes
else
case $choix in
FIRST)
# first ones:
head -$npick dq_modes.amplitudes_all > dq_modes.amplitudes
;;
RANDOM)
# nmod different ones:
nmodpic=0
while [ $nmodpic -lt $nmod ]
do
nmodpic=$(( $nmodpic + 1 ))
newmod=NO
while [ $newmod = NO ] ; do
numlin=$( awk -v nl=$nl 'BEGIN{srand(); print rand()*nl+1}' | awk -F '.' '{ print $1 }' )
modamp=`sed -n ${numlin}p dq_modes.amplitudes_all`
modnum=`echo $modamp | awk '{ print $1 }'` 
if [ $nmodpic -gt 1 ] ; then
   nfound=`awk '{ print $1 }' dq_modes.amplitudes | grep -c $modnum`
   if [ $nfound -eq 0 ] ; then newmod=YES ; fi
else
   newmod=YES
fi
done
echo $modamp >> dq_modes.amplitudes
done
;;
esac
fi

# ---------
# Main job:
# ---------

cp dqmod.dat_main dqmod.dat
echo 'SEED =' $seed >> dqmod.dat
# echo 'NUMBer of vectors read =' >> dqmod.dat
rm -f conformer_dqmod.pdb

./$execur > dq_modes.out

if grep -q 'Normal end' dq_modes.out ; then
nconf=$(( $nconf + 1 ))
[[ $nconf -eq $fconf ]] && egrep '\-Wn>|\-Er' dq_modes.out  

# Save conformers:
if [ $mpdb != NONE ] ; then
   [[ $nconf -eq $fconf ]] && awk '{ if ( substr($1,1,6) == "REMARK" ) 
   print }' conformer_dqmod.pdb | grep -v 'Amplitude of mode' > $mpdb
   echo 'MODEL' $nconf >> $mpdb
   awk '{ if ( $1 == "ATOM" || substr($1,1,6) == "HETATM" ) print }' conformer_dqmod.pdb >> $mpdb
   echo 'ENDMDL' >> $mpdb
   rm conformer_dqmod.pdb
else
   pdbout=$prefout'_conf'$nconf'.pdb'
   mv conformer_dqmod.pdb $pdbout
fi

if grep -q 'after perturbation' dq_modes.out ; then
   rmsper=`grep 'after perturbation' dq_modes.out | awk -F'=' '{ print $2 }'`
else
   rmsper=-1
fi
if grep -q 'after MINImization' dq_modes.out ; then
   rmsmin=`grep 'after MINImization' dq_modes.out | awk -F'=' '{ print $2 }'`
else
   rmsmin=-1
fi
if grep -q '<d>' dq_modes.out ; then
   dcamin=`grep '<d>' dq_modes.out | tail -1 | awk '{ print $6 }'`
   dcamax=`grep '<d>' dq_modes.out | tail -1 | awk '{ print $8 }'`
else
   dcamin=-1
   dcamax=-1
fi
if [ $mini -gt 0 ] ; then
echo $amplfil 'Try:' $try 'Conformer:' $nconf 'Rmsd:' $rmsper 'After-minimization:' $rmsmin 'CA-CA:' $dcamin $dcamax >> $pgmsum
else
echo $amplfil 'Try:' $try 'Conformer:' $nconf 'Rmsd:' $rmsper 'CA-CA:' $dcamin $dcamax >> $pgmsum
fi

if grep -q 'New SEED'  dq_modes.out ; then
   seed=`grep 'New SEED' dq_modes.out | awk -F'=' '{ print $2 }'`
fi
# Pb with dq_modes:
else
echo $pgmwrn 'Aborted.'
if [ $nconf -eq 0 ] ; then
if grep -q 'Coordinate statistics' dq_modes.out ; then
if grep -q 'Vector normalization'  dq_modes.out ; then
echo $pgmnam 'Coordinate and normal mode files are OK, though.'
else
echo ' '
echo $pgmerr 'Wrong normal mode file.'
exit
fi
else
echo ' '
echo $pgmerr 'Wrong coordinate file.'
exit
fi
fi
egrep '\-Wn>|\-Er' dq_modes.out
cp dq_modes.out dq_modes.out_lastpb
fi

# Next try:
done

# Pb with amplitude file:
else
echo $pgmwrn 'Wrong amplitude file.'
fi
else
echo $pgmwrn 'File not found.'
fi
# Next filename:
shift
nargs=$(( $nargs + 1 ))
done
echo $pgmnam $nargrd 'amplitude filename(s) considered.'
echo $pgmnam $nampl  'amplitude file(s) used.'
[[ $nred -gt 0 ]] && echo $pgmnam $nred 'of them being considered only once, since amplitude fluctuations (-fluct) were not specified.'

ngen=$(( $nconf - $fconf + 1 ))
echo $pgmnam $ngen 'conformation(s) generated.'
if [ $ngen -eq 0 ] ; then
echo ' '
echo $pgmerr 'Not much.'
exit
fi
[[ $mpdb != NONE ]] && echo $pgmnam 'Saved in PDB file "'$mpdb'".'

# Statistics:
# ===========

nstat=`grep 'Rmsd:' $pgmsum | awk '{ if ( NF > 6 ) print }' | wc -l | awk '{ print $1 }'` 
if [ $nstat -eq 0 ] ; then 
  echo ' '
  echo $pgmerr 'No rmsd record can be read by awk in file' $pgmsum
  exit 
else
  nstatmin=`grep 'Rmsd:' $pgmsum | grep -c 'minimization:'` 
fi

nbstat=`grep 'Rmsd:' $pgmsum | awk '{ sum+=$7 ; sumsq+=$7*$7 } END { sum=sum/NR ; sumsq=sqrt(sumsq/NR - sum^2) } END { print sum, "+/-", sumsq }'` 
echo ' '
echo $pgmnam 'Rmsd after perturbation for' $nstat 'conformer(s):' $nbstat
if [ $nstatmin -gt 0 ] ; then
nbstat=`grep 'Rmsd:' $pgmsum | awk '{ sum+=$9 ; sumsq+=$9*$9 } END { sum=sum/NR ; sumsq=sqrt(sumsq/NR - sum^2) } END { print sum, "+/-", sumsq }'` 
echo $pgmnam 'Rmsd after minimization for' $nstat 'conformer(s):' $nbstat
fi

# Cleaning:
# =========

if [ $debug = NO ] ; then 
rm -f dqmod.dat_main dq_modes.amplitudes_all dq_modes.amplitudes dqmod.amplitudes conformer.ener 
fi

echo ' '
echo $pgmnam 'Normal end.'
exit
