#!/bin/bash 
#CoM====================================================================
#CoM...Enm_modes.sh: Normal modes of an Elastic Network Model.
#CoM--------------------------------------------------------------------
#CoM
#CoM   Main goal: 
#CoM   ++++++++++
#CoM
#CoM  *To build and Elastic Network Model (ENM) and to compute
#CoM   the low-frequency normal modes of a set of systems.
#CoM
#CoM   Properties:
#CoM   +++++++++++
#CoM
#CoM  *Input coordinate files, in PDB or in FREE format.
#CoM
#CoM  *In PDB coordinate files, all ATOM and HETATM lines are taken
#CoM   into account. 
#CoM
#CoM  *Output eigenvector (mode) filenames are built with input ones.
#CoM   They are written in current directory.
#CoM
#CoM  *Uses programs pdbmat and diagstd or diagrtb.
#CoM
#CoM  *Source (fortran) programs are compiled in current directory.
#CoM   Purpose: to be able to reproduce the analysis in the future.
#CoM
#CoM   Attention : 
#CoM   +++++++++++
#CoM
#CoM  *Uses temporary files. 
#CoM   So, DO NOT run this script twice at a time in same directory !
#CoM
#CoM--------------------------------------------------------------------
#CoM   This software is released under the CeCILL FREE SOFTWARE LICENSE.
#CoM   In short: this license is compatible with the GNU GPL.
#CoM   Specifically: it grants users the right to modify and redistribute 
#CoM   this software within the framework of an open source distribution 
#CoM   model. The complete text of the license can be found there:
#CoM   http://www.cecill.info/licences/Licence_CeCILL_V2-en.html
#CoM-----------------------------------------------------------------------
#CoM   Please mail bug reports to yves-henri.sanejouand at univ-nantes(fr).
# v1.00: YHS-jan-2016 Nantes.
version='v1.12, April 2017.'

# Location (path) of fortran programs:
# ------------------------------------
pgmdir=~/inputs/shell/fortran

# Fortran compiler used (for instance, under Ubuntu: gfortran):
# -------------------------------------------------------------
gfor=gfortran

# Fortran programs and versions used hereafter:
# ---------------------------------------------
# Hessian building:
tirmat='pdbmat'
vrstir='3.106'

# Diagonalization.
# Standard (for small systems):
diamat='diagstd'
vrsdia='1.22'

# RTB:
rtbmat='diagrtb'
vrsrtb='2.67'

pgmprf='Enmmodes'
pgmnam='+'$pgmprf'> '
pgmwrn='%'$pgmprf'-Wn> '
pgmerr='%'$pgmprf'-Er> '

echo $pgmnam 'Computes low-frequency modes of Elastic Network Models.'
echo $pgmnam $version

# Arguments and options are read:
# -------------------------------

# Defauts:
nvecreq=106
rcut=NONE
kfce=1.0
mass=CONS
meth=RTB
out=CERFACS
debug=NO
prtl=0

# une liste de fichiers pour finir:
lstfil=NONE
if [ $# -gt 0 ] 
then
help=OFF
nargrd=$#
nargs=0
while [ $nargs -lt $nargrd ]
do
# Un - en premier caractere...
  cmd=`echo '+'$1 | tr '[A-Z]' '[a-z]' | sed 's/+//'`
  case $cmd in 
 '-h'|'-help') 
   help=ON 
   nargs=$(( $nargs + 1 ))
  ;;
  *)
  if [ $# -ge 2 ] ; then
  nargs=$(( $nargs + 2 ))
  fcararg=`echo $2 | cut -c1`
  if [ $fcararg = \- ] ; then
  case $2 in
  *[!\-,!\.,!0-9]*)
  echo ' '
  echo $pgmerr $cmd 'must be followed by a parameter (or by a number).'
  lstfil=FOUND
  nargs=$nargrd
  help=ON
  esac
  fi
  case $cmd in
 '-f')   
  nfil=$(( $# - 1 ))
  nargs=$nargrd
  lstfil=FOUND
  ;;
 '-c'|'-cut'|'-cutof'|'-cutoff') rcut=$2 ;;
 '-com'|'-comp'|'-compil'|'-compiler') gfor=$2 ;;
 '-con'|'-cons'|'-constant') kfce=$2 ;;
 '-d'|'-deb'|'-debug') debug=$2 ;;
 '-for'|'-fort'|'-fortran') pgmdir=$2 ;;
 '-k')  kfce=$2 ;;
 '-mas'|'-mass'|'-masses')  mass=$2 ;;
 '-met'|'-meth'|'-method')  meth=$2 ;;
 '-m'|'-mod'|'-mode'|'-modes') nvecreq=$2 ;;
 '-o'|'-out'|'-outp'|'-output') out=$2 ;;
 '-p'|'-prt'|'-pri'|'-print') prtl=$2 ;;
 '-sou'|'-source') pgmdir=$2 ;;
 '-v'|'-vec'|'-vect'|'-vector'|'-vectors') nvecreq=$2 ;;
  *) 
  echo ' '
  echo $pgmerr $cmd 'is not an expected option.'
  lstfil=FOUND
  nargs=$nargrd
  help=ON
  ;;
  esac
# Argument suivant:
  if [ $lstfil = NONE ] ; then shift ; fi
  else
  echo ' '
  echo $pgmerr $cmd 'is the last argument ?'
  nargs=$(( $nargs + 1 ))
  help=ON
  fi
  ;;
  esac
# Argument suivant:
  shift 
done
# Pas d'argument:
else
help=ON
fi

if [ $lstfil = NONE -a $help = OFF ] ; then
echo ' '
echo $pgmerr '-f filename(s): required.'
help=ON
fi

if [ $help = ON ] ; then
pgmout=`echo $pgmprf | tr '[A-Z]' '[a-z]'`
# -e ou \n pas assez standard ? (2015)
echo ' '
echo ' Action: Computes low-frequency modes of Elastic Network Models.'
echo ' Usage : '$pgmout'.sh { arguments } { options } -f filename(s)'      
echo ' '
echo ' Last argument:'
echo '-file    coordinate-filename(s)                                                  (mandatory)'
echo ' '
echo ' Main arguments:'
echo '-cutoff  cutoff-distance-for-building-the-Elastic-Network-Model                  (default:' $rcut')'
echo ' '
echo ' Other options:'
echo '-vectors number-of-modes-required                                                (default:' $nvecreq')'
echo '-constant for-the-harmonic-springs                                               (default:' $kfce')'
echo '-masses  of-atoms                                                                (default:' $mass')'
echo '-method  used-for-diagonalization                                                (default:' $meth')'
echo '-output  eigenvector-format                                                      (default:' $out')'
echo '-compile fortran-compiler                                                        (default:' $gfor')'
echo '-source  path-to-fortran-sources                                                 (default:' $pgmdir')'
echo '-print   level-for-outputs                                                       (default:' $prtl')'
echo ' '
exit
fi

# Checks:
# =======
pb=NO

metup=`echo $meth | tr '[a-z]' '[A-Z]'`
case $metup in
RTB) 
meth=$metup 
[[ $rcut = NONE ]] && rcut=5.0
;;
ST|STA|STAN|STAND|STANDA|STANDAR|STANDARD) 
meth=STANDARD 
[[ $rcut = NONE ]] && rcut=12.0
;;
*) 
  echo ' '
  echo $pgmerr '-method: valid diagonalisation method name required (STANDARD or RTB).'
  pb=YES ;;
esac

case $rcut in
*[!0-9,!\.]*) 
  echo ' '
  echo $pgmerr '-cutoff: positive real number required (distance Cutoff value).'
  pb=YES
esac

vecopt=$( echo $nvecreq | tr '[a-z]' '[A-Z]' )
case $vecopt in
ALL) ;; 
*[!0-9,!\-]*) 
  echo ' '
  echo $pgmerr '-vectors: integer number required (number of Vectors).'
  pb=YES
esac

case $kfce in
*[!0-9,!\.]*) 
  echo ' '
  echo $pgmerr '-constant: positive real number required (force-constant value).'
  pb=YES
esac

case $prtl in
*[!0-9,!\-]*) 
  echo $pgmwrn '-print: integer number required (printing level).'
  prtl=0
esac

outup=`echo $out | tr '[a-z]' '[A-Z]'`
case $outup in
CERF|CERFA|CERFACS) out=CERFACS ;;
MOD|MODE|MODES) out=MODES ;;
*) 
  echo ' '
  echo $pgmerr '-output: valid output eigenvector format required (CERFACS or MODES).'
  pb=YES ;;
esac

debup=`echo $debug | tr '[a-z]' '[A-Z]'`
case $debup in
Y|YES|O|OUI|T|TRUE) 
  debug=YES ;;
N|NO|NON|F|FALSE)
  debug=NO ;;
*) 
  echo $pgmwrn '-debug: boolean required (YES or NO).' ;;
esac

if [ $pb = YES ] ; then exit ; fi

# Compiling:
# ==========
# Compilation des fortrans (si necessaire):

# Pdbmat:
# -------
fornam=$tirmat
forvrs=$vrstir

# Full fortran program name:
vforpgm=$fornam'.f_v'$forvrs
[[ ! -f $vforpgm ]] && vforpgm=$pgmdir/$fornam'.f_v'$forvrs

forcur=$fornam'_vcur.f'
execur=$fornam'_vcur'
ndiff=-1
if [ -f $vforpgm -a -f $forcur ] ; then
ndiff=`diff -s $vforpgm $forcur | grep -c '<'`
if [ $ndiff -gt 0 ] ; then
   echo $pgmwrn '"'$forcur'" and "'$vforpgm'" differ. Former one replaced (a copy is saved).'
   cp $forcur $forcur'_old'
   cp $vforpgm $forcur
   rm -f $execur
fi
else
if [ -f $forcur ] ; then
   echo $pgmwrn 'Fortran source "'$vforpgm'" (see top of script) not found.'
   echo $pgmwrn 'Local fortran source to be used without checking if it is the expected one.'
fi
if [ -f $vforpgm ] ; then
   echo $pgmwrn 'Fortran source "'$vforpgm'" copied locally.'
   cp $vforpgm $forcur
   ndiff=0
fi
fi

if [ -x $execur ] ; then
echo $pgmwrn 'Local executable "'$execur'" to be used.'
if [ $ndiff -gt 0 ] ; then 
   echo $pgmwrn 'It is expected to have been compiled using "'$vforpgm'".' 
else
   if [ -f $vforpgm ] ; then echo $pgmwrn 'It may have been compiled using "'$vforpgm'"' ; fi
fi
else
echo $pgmnam 'Where is the fortran compiler ?'
if ! which $gfor ; then
   echo ' '
   echo $pgmerr $gfor 'fortran compiler needed (see top of script).'
   exit
fi
if [ -f $forcur ] 
then
   echo $pgmnam 'Fortran compilation to be performed locally.'
   vcur=`grep 'version=' $forcur | head -1 | sed 's/version=//' | sed 's/ Vers/vers/'`
   echo $pgmnam 'To be compiled:' $vcur
   $gfor -o $execur $forcur
else
   echo ' '
   echo $pgmerr 'No local executable.'
   echo $pgmerr 'Fortran source "'$vforpgm'" required (see top of script).'
   exit
fi
if [ -x $execur ] ; then
   echo $pgmnam 'Local executable' $execur 'built in current directory.'
else
   echo ' '
   echo $pgmerr 'Compilation of' $forcur 'failed.'
   exit
fi
fi
exemat=$execur

fordat=$fornam'.dat'
cat > $fordat << EOFmat 
COORdinate FILENAME        = structure.pdb
MATRIx FILENAME            = structure.sdijf
SAVED coordinate FILENAME  = structure.xyzm
INTERACtion DISTance CUTOF = $rcut
INTERACtion FORCE CONStant = $kfce
MASS values                = $mass
LevelSHIFT                 = 0.0001
Output PRINTing level      = $prtl
EOFmat

# Diagrtb:
# --------
case $meth in
STANDARD)
fornam=$diamat
forvrs=$vrsdia ;;
RTB)
fornam=$rtbmat
forvrs=$vrsrtb ;;
esac

# Full fortran program name:
vforpgm=$fornam'.f_v'$forvrs
[[ ! -f $vforpgm ]] && vforpgm=$pgmdir/$fornam'.f_v'$forvrs

forcur=$fornam'_vcur.f'
execur=$fornam'_vcur'
ndiff=-1
if [ -f $vforpgm -a -f $forcur ] ; then
ndiff=`diff -s $vforpgm $forcur | grep -c '<'`
if [ $ndiff -gt 0 ] ; then
   echo $pgmwrn '"'$forcur'" and "'$vforpgm'" differ. Former one replaced (a copy is saved).'
   cp $forcur $forcur'_old'
   cp $vforpgm $forcur
   rm -f $execur
fi
else
if [ -f $forcur ] ; then
   echo $pgmwrn 'Fortran source "'$vforpgm'" (see top of script) not found.'
   echo $pgmwrn 'Local fortran source to be used without checking if it is the expected one.'
fi
if [ -f $vforpgm ] ; then
   echo $pgmwrn 'Fortran source "'$vforpgm'" copied locally.'
   cp $vforpgm $forcur
   ndiff=0
fi
fi

if [ -x $execur ] ; then
echo $pgmwrn 'Local executable "'$execur'" to be used.'
if [ $ndiff -gt 0 ] ; then 
   echo $pgmwrn 'It is expected to have been compiled using "'$vforpgm'".' 
else
   if [ -f $vforpgm ] ; then echo $pgmwrn 'It may have been compiled using "'$vforpgm'"' ; fi
fi
else
if [ -f $forcur ] 
then
   echo $pgmnam 'Fortran compilation to be performed locally.'
   vcur=`grep 'version=' $forcur | head -1 | sed 's/version=//' | sed 's/ Vers/vers/'`
   echo $pgmnam 'To be compiled:' $vcur
   $gfor -o $execur $forcur
else
   echo ' '
   echo $pgmerr 'No local executable.'
   echo $pgmerr 'Fortran source "'$vforpgm'" required (see top of script).'
   exit
fi
if [ -x $execur ] ; then
   echo $pgmnam 'Local executable' $execur 'built in current directory.'
else
   echo ' '
   echo $pgmerr 'Compilation of' $forcur 'failed.'
   exit
fi
fi

case $meth in
STANDARD)
cat > enm_modes.head_diag << EOFstd
structure.sdijf
n
EOFstd
diaginp=$fornam'.inp'
exestd=$execur ;;
RTB) 
cat > enm_modes.head_diag << EOFrtb 
COORdinate filename         = structure.xyzm
MATRIx FILENAME             = structure.sdijf
Eigenvector OUTPut filename = structure.eigenfacs
Nb of residues per BLOck    = 1
Origin of MASS values       = $mass
Output PRINting level       = $prtl
EOFrtb
fordat=$fornam'.dat'
exertb=$execur ;;
esac

pgmsum=`echo $pgmprf | tr '[A-Z]' '[a-z]'`
pgmsum=$pgmsum'.sum'
if [ -f $pgmsum ]
then
  echo $pgmwrn $pgmsum 'file already exists. A copy is saved.'
  mv $pgmsum $pgmsum'_old'
fi
# Single column title (for awk, gnuplot, etc):
echo $pgmnam $version | sed 's/ //g' > $pgmsum
echo $pgmnam 'Job started on' `date` ',in' $HOST 'directory' `pwd` | sed 's/ //g' >> $pgmsum
echo ' PDB, Atoms, Residues, Non-zero matrix elements, Vectors, Eigenvalues 7, 8, 9 ' >> $pgmsum

# Loop on input files(s):
# =======================

# Check: only files among arguments left ? 
if [ `ls $* 2>&1 | grep -c 'ls:'` -eq 0 ] ; then
   nargs=$#
else
   echo ' '
   echo $pgmerr 'Last arguments are ALL expected to be valid filename(s).'
   exit
fi
echo ' '
echo $pgmnam $nfil 'filename(s) to be considered.'

diag=NO
if [ $vecopt = ALL ] ; then
   diag=OK
else
   [[ $nvecreq -gt 0 ]] && diag=OK
fi
if [ $diag = OK ] ; then
echo $pgmnam $nvecreq 'modes to be computed with the' $meth 'method and an ENM distance-cutoff of' $rcut '(Angs).'
else
echo $pgmnam 'Hessian to be computed for each structure, with an ENM distance-cutoff of' $rcut '(Angs).'
fi
echo ' '

rcutout=`echo $rcut | sed 's/\./p/'`
nargrd=$nargs
nargs=0
nhes=0
nmod=0
while [ $nargs -lt $nargrd ]
do
echo $pgmnam 'File: "'$1'"' 
cp $1 structure.pdb
echo $pgmnam 'Hessian matrix to be set up.'

rm -f structure.sdijf structure.xyzm
./$exemat > enm_modes.out_matrix

if [ -f structure.sdijf -a -f structure.xyzm ] ; then
   nelmt=`awk '{ if ( $3 == "non-zero" && $4 == "elements." ) print $2 }' enm_modes.out_matrix`
   if [ ${#nelmt} -eq 0 ] ; then nelmt=0 ; fi
else
   nelmt=0
fi

if [ $nelmt -gt 0 ] ; then
nat=`grep 'Number of atoms found'    enm_modes.out_matrix | awk -F'=' '{ if ( NF == 2 ) print $2 ; else print "0" }'`
nrs=`grep 'Number of residues found' enm_modes.out_matrix | awk -F'=' '{ if ( NF == 2 ) print $2 ; else print "0" }'`
echo $pgmnam $nat 'atoms taken into account.'
echo $pgmnam $nrs 'residues taken into account.'
grep 'non-zero elements' enm_modes.out_matrix
natmin=$(( 3*$nrs ))
if [ $meth = RTB ] ; then
if [ $nat -lt $natmin ] ; then
   echo $pgmwrn 'Not enough atoms per residue for the RTB method.'
   nelmt=0
fi
fi
fi

if [ $nelmt -gt 0 ] ; then
nhes=$(( $nhes + 1 ))
if [ $diag = OK ] ; then
echo $pgmnam 'Hessian matrix to be diagonalized.'

case $vecopt in
ALL)
[[ $meth = STANDARD ]] && nvec=$(( 3*$nat )) 
[[ $meth = RTB ]] && nvec=$(( 3*$nrs )) 
;;
*)
nvec=$nvecreq ;;
esac

rm -f structure.eigenfacs
case $meth in 
STANDARD)
cp enm_modes.head_diag $diaginp
echo $nvec >> $diaginp

./$exestd < $diaginp > enm_modes.out_diag 
;;
RTB) 
cp enm_modes.head_diag $fordat
echo 'Nb of VECTors required      =' $nvec >> $fordat

./$exertb > enm_modes.out_diag 
;;
esac

if [ -f structure.eigenfacs ] ; then
   nvecout=`grep -c VECTOR structure.eigenfacs`
else
   nvecout=0
fi

if [ $nvecout -gt 0 ] ; then
if [ $nvecout -ne $nvec ] ; then echo $pgmwrn $nvecout 'modes obtained.' ; fi
egrep -A2 'Selected eigenvalues:|Eigenvalues:' enm_modes.out_diag

case $meth in
STANDARD) suffx='_r'$rcutout'_'$nvecout'vec' ;;
RTB) suffx='_r'$rcutout'rtb_'$nvecout'vec' ;;
esac

vecfil=`basename $1`
pref=`echo $vecfil | awk -F'.' '{ print $1 }'`
if [ ${#pref} -gt 0 ] ; then
   nomvec=${pref}${suffx}
else
   nomvec=${vecfil}${suffx}
fi

if [ $out = CERFACS ] ; then
   nomvec=$nomvec'.eigenfacs'
   cp structure.eigenfacs $nomvec
else
   nomvec=$nomvec'.modes'
   echo '!' $pgmnam $version > $nomvec
   if grep -q 'Coordinate file in PDB format' enm_modes.out_matrix ; then
   echo 'MODEL 1'           >> $nomvec
   awk '{ if ( $1 == "ATOM" || substr($1,1,6) == "HETATM" ) print }' structure.pdb >> $nomvec
   echo 'ENDMDL'            >> $nomvec
   echo 'MODEL 2'           >> $nomvec
   else
   echo 'MODEL'             >> $nomvec
   fi
   cat structure.xyzm       >> $nomvec
   echo 'ENDMDL'            >> $nomvec
   echo 'MODES'             >> $nomvec
   cat structure.eigenfacs  >> $nomvec
   echo 'ENDMOD'            >> $nomvec
fi

eig7=$( awk '{ if ( $1 == "VECTOR" && $2 == "7" ) print $4 }' structure.eigenfacs )
eig8=$( awk '{ if ( $1 == "VECTOR" && $2 == "8" ) print $4 }' structure.eigenfacs )
eig9=$( awk '{ if ( $1 == "VECTOR" && $2 == "9" ) print $4 }' structure.eigenfacs )

echo $pgmnam 'File "'$nomvec'" saved.'
nmod=$(( $nmod + 1 ))
else

echo $pgmwrn 'Hessian matrix could not be diagonalized.'
cp enm_modes.out_diag enm_modes.out_diag_lastpb
nvecout=0
eig7=NONE
eig8=NONE
eig9=NONE

fi
fi
echo $1"," $nat"," $nrs"," $nelmt"," $nvecout"," $eig7"," $eig8"," $eig9 >> $pgmsum
else
echo $pgmwrn 'Hessian matrix could not be produced.'
cp enm_modes.out_matrix enm_modes.out_matrix_lastpb
fi
# Next filename:
shift
nargs=$(( $nargs + 1 ))
done
echo ' '
echo $pgmnam $nhes 'Hessian(s) obtained.'
echo $pgmnam $nmod 'set(s) of normal modes saved.'

# Statistics:
# -----------
if [ $nhes -eq 0 ] ; then 
   echo $pgmwrn 'Not much.'
   exit
else
   nstat=$nhes
fi
nbstat=`tail -n +${nhes} $pgmsum | awk -F',' -v n=$nhes '{ sum+=$4 ; sumsq+=$4*$4 } END { sum=sum/n ; sumsq=sqrt(sumsq/n - sum^2) } END { print sum, "+/-", sumsq }'` 
echo ' '
echo $pgmnam 'Statistics for' $nstat 'points:' $nbstat 'non-zero matrix elements.'

# Cleaning:
if [ $debug = NO ] ; then rm -f structure.pdb structure.sdijf structure.xyzm structure.eigenfacs enm_modes.head_diag ; fi

echo ' '
echo $pgmnam 'Normal end.'
exit
